<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * HugOffers API Helper
 * Handles API communication with HugOffers platform
 */

if (!function_exists('get_hugoffers_api_data')) {
    /**
     * Fetch data from HugOffers API
     * 
     * @param string $api_key The API key for HugOffers
     * @param int $page Page number for pagination
     * @return array API response data or empty array on failure
     */
    function get_hugoffers_api_data($api_key, $page = 1) {
        // HugOffers API endpoint based on documentation
        $api_url = "http://aititude.hoapi0.com/v1?cid=aititude&token=" . urlencode($api_key) . "&page=100," . intval($page);
        
        // Initialize cURL
        $ch = curl_init();
        
        // Set cURL options
        curl_setopt($ch, CURLOPT_URL, $api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_USERAGENT, 'CoinZi-HugOffers-Client/1.0');
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Accept: application/json'
        ));
        
        // Execute the request
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        // Check for cURL errors
        if ($error) {
            log_message('error', 'HugOffers API cURL Error: ' . $error);
            return array();
        }
        
        // Check HTTP response code
        if ($http_code !== 200) {
            log_message('error', 'HugOffers API HTTP Error: ' . $http_code . ' - Response: ' . $response);
            return array();
        }
        
        // Decode JSON response
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            log_message('error', 'HugOffers API JSON Error: ' . json_last_error_msg());
            return array();
        }
        
        return $data;
    }
}

if (!function_exists('get_hugoffers_offer_by_id')) {
    /**
     * Get single offer by campaign ID from API
     * 
     * @param string $api_key The API key for HugOffers
     * @param string $campaign_id Campaign ID to fetch
     * @return array|false Offer data or false if not found
     */
    function get_hugoffers_offer_by_id($api_key, $campaign_id) {
        $api_data = get_hugoffers_api_data($api_key);
        
        if (isset($api_data['success']) && $api_data['success'] && isset($api_data['offers'])) {
            foreach ($api_data['offers'] as $offer) {
                if (isset($offer['campid']) && $offer['campid'] == $campaign_id) {
                    return format_hugoffers_offer($offer);
                }
            }
        }
        
        return false;
    }
}

if (!function_exists('format_hugoffers_offer')) {
    /**
     * Format HugOffers API offer data to internal format
     * 
     * @param array $offer Raw offer data from API
     * @return array Formatted offer data
     */
    function format_hugoffers_offer($offer) {
        // Extract images from creatives_link (handle JSON string)
        $images = array();
        if (isset($offer['creatives_link'])) {
            $creatives = $offer['creatives_link'];
            
            // If it's a JSON string, decode it
            if (is_string($creatives)) {
                $creatives = json_decode($creatives, true);
            }
            
            if (is_array($creatives)) {
                foreach ($creatives as $creative) {
                    if (isset($creative['url'])) {
                        $images[] = array(
                            'url' => $creative['url'],
                            'width' => isset($creative['width']) ? intval($creative['width']) : 0,
                            'height' => isset($creative['height']) ? intval($creative['height']) : 0
                        );
                    }
                }
            }
        }
        
        // Extract videos from video_link (handle JSON string)
        $videos = array();
        if (isset($offer['video_link'])) {
            $videodata = $offer['video_link'];
            
            // If it's a JSON string, decode it
            if (is_string($videodata)) {
                $videodata = json_decode($videodata, true);
            }
            
            if (is_array($videodata)) {
                foreach ($videodata as $video) {
                    if (isset($video['url'])) {
                        $videos[] = array(
                            'url' => $video['url'],
                            'duration' => isset($video['duration']) ? intval($video['duration']) : 0,
                            'width' => isset($video['width']) ? intval($video['width']) : 0,
                            'height' => isset($video['height']) ? intval($video['height']) : 0
                        );
                    }
                }
            }
        }
        
        return array(
            'campid' => isset($offer['campid']) ? $offer['campid'] : '',
            'offer_name' => isset($offer['offer_name']) ? $offer['offer_name'] : 'Unknown Offer',
            'status' => isset($offer['status']) ? $offer['status'] : 'unknown',
            'platform' => isset($offer['platform']) ? $offer['platform'] : 'All',
            'tracking_link' => isset($offer['tracking_link']) ? $offer['tracking_link'] : '',
            'geo' => isset($offer['geo']) ? $offer['geo'] : 'ALL',
            'preview_link' => isset($offer['preview_link']) ? $offer['preview_link'] : '',
            'price' => isset($offer['price']) ? floatval($offer['price']) : 0.0,
            'price_model' => isset($offer['price_model']) ? $offer['price_model'] : 'cpi',
            'app_category' => isset($offer['app_category']) ? $offer['app_category'] : 'General',
            'daily_cap' => isset($offer['daily_cap']) ? intval($offer['daily_cap']) : 0,
            'icon_link' => isset($offer['icon_link']) ? $offer['icon_link'] : '',
            'images' => $images,
            'videos' => $videos,
            'app_name' => isset($offer['app_name']) ? $offer['app_name'] : '',
            'app_desc' => isset($offer['app_desc']) ? $offer['app_desc'] : '',
            'app_size' => isset($offer['app_size']) ? $offer['app_size'] : '',
            'app_rate' => isset($offer['app_rate']) ? floatval($offer['app_rate']) : 0.0,
            'app_id' => isset($offer['app_id']) ? $offer['app_id'] : '',
            'requirement_traffic' => isset($offer['requirement_traffic']) ? $offer['requirement_traffic'] : ''
        );
    }
}

if (!function_exists('log_hugoffers_conversion')) {
    /**
     * Log HugOffers conversion for tracking
     * 
     * @param string $click_id The click ID
     * @param string $offer_id The offer ID
     * @param float $payout The payout amount
     * @param int $user_id The user ID
     * @return bool Success status
     */
    function log_hugoffers_conversion($click_id, $offer_id, $payout, $user_id) {
        $CI =& get_instance();
        
        // Log conversion details
        $log_data = array(
            'click_id' => $click_id,
            'offer_id' => $offer_id,
            'payout' => $payout,
            'user_id' => $user_id,
            'timestamp' => time(),
            'ip_address' => $CI->input->ip_address(),
            'user_agent' => $CI->input->user_agent()
        );
        
        log_message('info', 'HugOffers Conversion: ' . json_encode($log_data));
        
        return true;
    }
}

if (!function_exists('validate_hugoffers_postback')) {
    /**
     * Validate HugOffers postback signature
     * 
     * @param array $params Postback parameters
     * @param string $secret_key Secret key for validation
     * @return bool Validation result
     */
    function validate_hugoffers_postback($params, $secret_key) {
        // Implement signature validation if HugOffers provides one
        // For now, we'll validate based on required parameters
        
        $required_params = array('aff_sub', 'payout');
        
        foreach ($required_params as $param) {
            if (!isset($params[$param]) || empty($params[$param])) {
                return false;
            }
        }
        
        return true;
    }
} 